<?php
// profile.php
require_once 'connect.php';
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

if(!isset($_SESSION["logged_in"]) || $_SESSION["logged_in"] !== true) {
    header("location: login.php");
    exit();
}

$user_id = $_SESSION['user_id'];

// Fetch user data
$stmt = $conn->prepare("SELECT username, mobile_number, Balancer, profile_picture, nid_front, nid_back, created_at, is_verified FROM user WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
$stmt->close();

$default_profile_pic = 'images/profile/userimg.jpeg';
$profile_picture = !empty($user['profile_picture']) ? $user['profile_picture'] : $default_profile_pic;
$nid_front = !empty($user['nid_front']) ? $user['nid_front'] : 'images/nid/default_front.png';
$nid_back = !empty($user['nid_back']) ? $user['nid_back'] : 'images/nid/default_back.png';

// Check if all documents are uploaded
$all_documents_uploaded = !empty($user['profile_picture']) && $user['profile_picture'] !== $default_profile_pic && !empty($user['nid_front']) && !empty($user['nid_back']);

// Update verification status if all documents are uploaded but not verified yet
if ($all_documents_uploaded && empty($user['is_verified'])) {
    $stmt = $conn->prepare("UPDATE user SET is_verified = 1 WHERE id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $stmt->close();
    
    // Set congratulation flag in session
    $_SESSION['show_congratulation'] = true;
    header("Location: profile.php");
    exit();
}

// Handle file uploads
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $target_dir = "uploads/";
    
    // Profile Picture Upload (only if still using default)
    if (isset($_FILES['profile_picture']) && $user['profile_picture'] === $default_profile_pic) {
        $target_dir_pp = $target_dir . "profile_pics/";
        if (!file_exists($target_dir_pp)) {
            mkdir($target_dir_pp, 0777, true);
        }
        
        $file_extension = pathinfo($_FILES["profile_picture"]["name"], PATHINFO_EXTENSION);
        $target_file = $target_dir_pp . "profile_" . $user_id . "." . $file_extension;
        
        // Check if file was actually uploaded
        if ($_FILES["profile_picture"]["error"] !== UPLOAD_ERR_OK) {
            $_SESSION['message'] = ['type' => 'error', 'text' => 'Error uploading file.'];
        } else {
            // Validate and move file
            $check = @getimagesize($_FILES["profile_picture"]["tmp_name"]);
            if($check === false) {
                $_SESSION['message'] = ['type' => 'error', 'text' => 'File is not an image.'];
            } elseif ($_FILES["profile_picture"]["size"] > 15000000) {
                $_SESSION['message'] = ['type' => 'error', 'text' => 'File is too large (Max 2MB).'];
            } elseif(!in_array(strtolower($file_extension), ['jpg', 'png', 'jpeg', 'gif'])) {
                $_SESSION['message'] = ['type' => 'error', 'text' => 'Only JPG, PNG, JPEG & GIF files are allowed.'];
            } elseif (move_uploaded_file($_FILES["profile_picture"]["tmp_name"], $target_file)) {
                $stmt = $conn->prepare("UPDATE user SET profile_picture = ? WHERE id = ?");
                $stmt->bind_param("si", $target_file, $user_id);
                if($stmt->execute()) {
                    $_SESSION['profile_picture'] = $target_file;
                    $_SESSION['message'] = ['type' => 'success', 'text' => 'Profile picture uploaded successfully. Waiting for verification.'];
                    header("Location: profile.php");
                    exit();
                } else {
                    $_SESSION['message'] = ['type' => 'error', 'text' => 'Database error.'];
                }
                $stmt->close();
            } else {
                $_SESSION['message'] = ['type' => 'error', 'text' => 'Error uploading file.'];
            }
        }
    }
    
    // NID Front Upload (only if not already uploaded)
    if (isset($_FILES['nid_front']) && empty($user['nid_front'])) {
        $target_dir_nid = $target_dir . "nid_docs/";
        if (!file_exists($target_dir_nid)) {
            mkdir($target_dir_nid, 0777, true);
        }
        
        $file_extension = pathinfo($_FILES["nid_front"]["name"], PATHINFO_EXTENSION);
        $target_file = $target_dir_nid . "nid_front_" . $user_id . "." . $file_extension;
        
        // Check if file was actually uploaded
        if ($_FILES["nid_front"]["error"] !== UPLOAD_ERR_OK) {
            $_SESSION['message'] = ['type' => 'error', 'text' => 'Error uploading NID front.'];
        } else {
            // Validate and move file
            $check = @getimagesize($_FILES["nid_front"]["tmp_name"]);
            if($check === false) {
                $_SESSION['message'] = ['type' => 'error', 'text' => 'NID front file is not an image.'];
            } elseif ($_FILES["nid_front"]["size"] > 15000000) {
                $_SESSION['message'] = ['type' => 'error', 'text' => 'NID front file is too large (Max 2MB).'];
            } elseif(!in_array(strtolower($file_extension), ['jpg', 'png', 'jpeg', 'gif'])) {
                $_SESSION['message'] = ['type' => 'error', 'text' => 'Only JPG, PNG, JPEG files are allowed for NID.'];
            } elseif (move_uploaded_file($_FILES["nid_front"]["tmp_name"], $target_file)) {
                $stmt = $conn->prepare("UPDATE user SET nid_front = ? WHERE id = ?");
                $stmt->bind_param("si", $target_file, $user_id);
                if($stmt->execute()) {
                    $_SESSION['message'] = ['type' => 'success', 'text' => 'NID front uploaded successfully. Waiting for verification.'];
                    header("Location: profile.php");
                    exit();
                } else {
                    $_SESSION['message'] = ['type' => 'error', 'text' => 'Database error.'];
                }
                $stmt->close();
            } else {
                $_SESSION['message'] = ['type' => 'error', 'text' => 'Error uploading NID front.'];
            }
        }
    }
    
    // NID Back Upload (only if not already uploaded)
    if (isset($_FILES['nid_back']) && empty($user['nid_back'])) {
        $target_dir_nid = $target_dir . "nid_docs/";
        if (!file_exists($target_dir_nid)) {
            mkdir($target_dir_nid, 0777, true);
        }
        
        $file_extension = pathinfo($_FILES["nid_back"]["name"], PATHINFO_EXTENSION);
        $target_file = $target_dir_nid . "nid_back_" . $user_id . "." . $file_extension;
        
        // Check if file was actually uploaded
        if ($_FILES["nid_back"]["error"] !== UPLOAD_ERR_OK) {
            $_SESSION['message'] = ['type' => 'error', 'text' => 'Error uploading NID back.'];
        } else {
            // Validate and move file
            $check = @getimagesize($_FILES["nid_back"]["tmp_name"]);
            if($check === false) {
                $_SESSION['message'] = ['type' => 'error', 'text' => 'NID back file is not an image.'];
            } elseif ($_FILES["nid_back"]["size"] > 15000000) {
                $_SESSION['message'] = ['type' => 'error', 'text' => 'NID back file is too large (Max 2MB).'];
            } elseif(!in_array(strtolower($file_extension),  ['jpg', 'png', 'jpeg', 'gif'])) {
                $_SESSION['message'] = ['type' => 'error', 'text' => 'Only JPG, PNG, JPEG files are allowed for NID.'];
            } elseif (move_uploaded_file($_FILES["nid_back"]["tmp_name"], $target_file)) {
                $stmt = $conn->prepare("UPDATE user SET nid_back = ? WHERE id = ?");
                $stmt->bind_param("si", $target_file, $user_id);
                if($stmt->execute()) {
                    $_SESSION['message'] = ['type' => 'success', 'text' => 'NID back uploaded successfully. Waiting for verification.'];
                    header("Location: profile.php");
                    exit();
                } else {
                    $_SESSION['message'] = ['type' => 'error', 'text' => 'Database error.'];
                }
                $stmt->close();
            } else {
                $_SESSION['message'] = ['type' => 'error', 'text' => 'Error uploading NID back.'];
            }
        }
    }
    
    header("Location: profile.php");
    exit();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Profile - Easy Loan</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        :root {
            --primary-color: #4e73df;
            --secondary-color: #2e59d9;
            --accent-color: #f6c23e;
            --light-color: #f8f9fc;
            --dark-color: #5a5c69;
            --success-color: #1cc88a;
            --danger-color: #e74a3b;
        }
        
        body {
            background-color: #f8f9fc;
            font-family: 'Nunito', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
        }
        
        .profile-container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .profile-header {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
            color: white;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 30px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            position: relative;
            overflow: hidden;
        }
        
        .profile-header::after {
            content: "";
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100" preserveAspectRatio="none"><path fill="rgba(255,255,255,0.1)" d="M0,0 L100,0 L100,100 L0,100 Z" /></svg>');
            opacity: 0.1;
            pointer-events: none;
        }
        
        #particles-js {
            position: absolute;
            width: 100%;
            height: 100%;
            top: 0;
            left: 0;
            z-index: 0;
        }
        
        .profile-header-content {
            position: relative;
            z-index: 1;
        }
        
        .profile-pic-container {
            position: relative;
            width: 150px;
            height: 150px;
            margin: 0 auto 20px;
            z-index: 1;
        }
        
        .profile-pic {
            width: 100%;
            height: 100%;
            object-fit: cover;
            border-radius: 50%;
            border: 5px solid white;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .profile-pic-upload {
            display: none;
        }
        
        .document-card {
            background-color: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            transition: all 0.3s ease;
            position: relative;
        }
        
        .document-card.completed {
            border-left: 5px solid var(--success-color);
        }
        
        .document-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .document-image {
            width: 100%;
            height: 200px;
            object-fit: contain;
            background-color: #f8f9fc;
            border: 1px dashed #ddd;
            border-radius: 5px;
            padding: 10px;
            margin-bottom: 15px;
        }
        
        .btn-upload {
            background-color: var(--primary-color);
            color: white;
            border: none;
            border-radius: 5px;
            padding: 8px 15px;
            font-size: 14px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .btn-upload:hover {
            background-color: var(--secondary-color);
            transform: translateY(-2px);
        }
        
        .btn-upload:disabled {
            background-color: #cccccc;
            cursor: not-allowed;
        }
        
        .user-info-card {
            background-color: white;
            border-radius: 10px;
            padding: 25px;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        
        .info-item {
            padding: 10px 0;
            border-bottom: 1px solid #eee;
        }
        
        .info-item:last-child {
            border-bottom: none;
        }
        
        .info-label {
            font-weight: 600;
            color: var(--dark-color);
        }
        
        .info-value {
            color: #6c757d;
        }
        
        .balance-display {
            font-size: 2rem;
            font-weight: 700;
            color: white;
            margin: 10px 0;
        }
        
        .alert-message {
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 1000;
            max-width: 400px;
        }
        
        @media (max-width: 768px) {
            .profile-pic-container {
                width: 120px;
                height: 120px;
            }
            
            .balance-display {
                font-size: 1.5rem;
            }
            
            .document-image {
                height: 150px;
            }
        }
        
        .upload-label {
            display: block;
            margin-bottom: 10px;
            font-weight: 600;
            color: var(--dark-color);
        }
        
        .upload-status {
            font-size: 12px;
            color: #6c757d;
            margin-top: 5px;
        }
        
        .status-badge {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .status-uploaded {
            background-color: #d4edda;
            color: #155724;
        }
        
        .status-pending {
            background-color: #fff3cd;
            color: #856404;
        }
        
        .verification-badge {
            position: absolute;
            top: 15px;
            right: 15px;
            font-size: 24px;
            z-index: 1;
        }
        
        .verified-badge {
            color: var(--success-color);
        }
        
        .not-verified-badge {
            color: var(--danger-color);
        }
        
        .congratulation-modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.8);
            z-index: 2000;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        
        .congratulation-content {
            background: white;
            padding: 40px;
            border-radius: 15px;
            text-align: center;
            max-width: 600px;
            width: 90%;
            animation: bounceIn 1s;
            position: relative;
            overflow: hidden;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
            border: 2px solid var(--accent-color);
        }
        
        .congratulation-content::before {
            content: "";
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, rgba(255,255,255,0) 70%);
            animation: shine 3s infinite;
            z-index: 0;
        }
        
        .congratulation-content h2 {
            color: var(--primary-color);
            margin-bottom: 20px;
            font-size: 2.2rem;
            position: relative;
            z-index: 1;
            text-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .congratulation-content p {
            font-size: 1.1rem;
            margin-bottom: 10px;
            position: relative;
            z-index: 1;
        }
        
        .congratulation-btn {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            border: none;
            color: white;
            padding: 12px 30px;
            font-size: 1.1rem;
            border-radius: 50px;
            margin-top: 20px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
            transition: all 0.3s ease;
            position: relative;
            z-index: 1;
            overflow: hidden;
        }
        
        .congratulation-btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 20px rgba(0,0,0,0.3);
            color: white;
        }
        
        .congratulation-btn::before {
            content: "";
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
            transition: 0.5s;
        }
        
        .congratulation-btn:hover::before {
            left: 100%;
        }
        
        .confetti {
            position: absolute;
            width: 10px;
            height: 10px;
            background-color: #f00;
            border-radius: 50%;
            animation: confetti-fall 5s linear infinite;
        }
        
        @keyframes confetti-fall {
            0% {
                transform: translateY(-100vh) rotate(0deg);
                opacity: 1;
            }
            100% {
                transform: translateY(100vh) rotate(360deg);
                opacity: 0;
            }
        }
        
        @keyframes shine {
            0% {
                transform: rotate(0deg);
            }
            100% {
                transform: rotate(360deg);
            }
        }
        
        .progress-container {
            margin: 20px 0;
        }
        
        .progress-bar {
            height: 10px;
            border-radius: 5px;
            background-color: #e9ecef;
            overflow: hidden;
        }
        
        .progress {
            height: 100%;
            background: linear-gradient(90deg, var(--primary-color), var(--secondary-color));
            transition: width 0.5s ease;
        }
        
        .progress-text {
            font-size: 12px;
            color: var(--dark-color);
            margin-top: 5px;
            text-align: right;
        }
        
        .document-status {
            display: flex;
            justify-content: space-between;
            margin-top: 15px;
        }
        
        .document-item {
            text-align: center;
            flex: 1;
        }
        
        .document-icon {
            font-size: 24px;
            margin-bottom: 5px;
        }
        
        .completed {
            color: var(--success-color);
        }
        
        .pending {
            color: var(--danger-color);
        }
        
        /* ID Review Modal Styles */
        .id-review-modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.9);
            z-index: 3000;
            justify-content: center;
            align-items: center;
        }
        
        .id-review-content {
            background: white;
            border-radius: 10px;
            width: 90%;
            max-width: 800px;
            max-height: 90vh;
            overflow: auto;
            padding: 20px;
            position: relative;
        }
        
        .id-review-close {
            position: absolute;
            top: 15px;
            right: 15px;
            font-size: 24px;
            color: #aaa;
            cursor: pointer;
        }
        
        .id-review-close:hover {
            color: #333;
        }
        
        .id-review-title {
            text-align: center;
            margin-bottom: 20px;
            color: var(--primary-color);
        }
        
        .id-review-images {
            display: flex;
            flex-wrap: wrap;
            gap: 20px;
            justify-content: center;
        }
        
        .id-review-image-container {
            flex: 1;
            min-width: 300px;
            text-align: center;
        }
        
        .id-review-image {
            max-width: 100%;
            max-height: 400px;
            object-fit: contain;
            border: 1px solid #ddd;
            border-radius: 5px;
        }
        
        .id-review-caption {
            margin-top: 10px;
            font-weight: bold;
            color: var(--dark-color);
        }
        
        .permanent-message {
            background-color: #f8f9fc;
            border-left: 4px solid var(--success-color);
            padding: 15px;
            border-radius: 5px;
            margin-top: 15px;
        }
        
        .checkmark-circle {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            width: 40px;
            height: 40px;
            background-color: var(--success-color);
            color: white;
            border-radius: 50%;
            margin-right: 10px;
        }
        
        /* Camera Modal Styles */
        .camera-modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.9);
            z-index: 3000;
            justify-content: center;
            align-items: center;
        }
        
        .camera-content {
            background: white;
            border-radius: 10px;
            width: 90%;
            max-width: 800px;
            padding: 20px;
            position: relative;
        }
        
        .camera-close {
            position: absolute;
            top: 15px;
            right: 15px;
            font-size: 24px;
            color: #aaa;
            cursor: pointer;
        }
        
        .camera-close:hover {
            color: #333;
        }
        
        .camera-title {
            text-align: center;
            margin-bottom: 20px;
            color: var(--primary-color);
        }
        
        .camera-preview {
            width: 100%;
            max-height: 500px;
            background-color: #000;
            margin-bottom: 15px;
            border-radius: 5px;
        }
        
        .camera-buttons {
            display: flex;
            justify-content: center;
            gap: 15px;
        }
        
        .camera-button {
            padding: 10px 20px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
            transition: all 0.3s;
        }
        
        .camera-capture {
            background-color: var(--primary-color);
            color: white;
        }
        
        .camera-capture:hover {
            background-color: var(--secondary-color);
        }
        
        .camera-switch {
            background-color: var(--dark-color);
            color: white;
        }
        
        .camera-switch:hover {
            background-color: #3d3f4d;
        }
        
        .camera-cancel {
            background-color: var(--danger-color);
            color: white;
        }
        
        .camera-cancel:hover {
            background-color: #c82333;
        }
        
        /* File Input Customization */
        .file-input-container {
            position: relative;
            overflow: hidden;
            display: inline-block;
            width: 100%;
        }
        
        .file-input-button {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            padding: 8px 15px;
            background-color: var(--primary-color);
            color: white;
            border-radius: 5px;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .file-input-button:hover {
            background-color: var(--secondary-color);
        }
        
        .file-input {
            position: absolute;
            left: 0;
            top: 0;
            opacity: 0;
            width: 100%;
            height: 100%;
            cursor: pointer;
        }
        
        .upload-options {
            display: flex;
            gap: 10px;
            margin-bottom: 15px;
        }
        
        .upload-option-btn {
            flex: 1;
            padding: 10px;
            text-align: center;
            border-radius: 5px;
            background-color: #f0f0f0;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .upload-option-btn:hover {
            background-color: #e0e0e0;
        }
        
        .upload-option-btn.active {
            background-color: var(--primary-color);
            color: white;
        }
    </style>
</head>
<body>
    <div class="profile-container">
        <!-- Alert Message -->
        <?php if(isset($_SESSION['message'])): ?>
            <div class="alert alert-<?= $_SESSION['message']['type'] ?> alert-dismissible fade show alert-message" role="alert">
                <?= $_SESSION['message']['text'] ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php unset($_SESSION['message']); ?>
        <?php endif; ?>
        
        <!-- Congratulation Modal -->
        <?php if(isset($_SESSION['show_congratulation']) && $_SESSION['show_congratulation']): ?>
        <div class="congratulation-modal animate__animated animate__fadeIn">
            <div class="congratulation-content animate__animated animate__bounceIn">
                <div style="position: relative; z-index: 2;">
                    <div class="checkmark-circle">
                        <i class="fas fa-check fa-2x"></i>
                    </div>
                    <h2>Congratulations! 🎉</h2>
                    <p>Your account has been successfully verified!</p>
                    <p>You now have full access to all features of our platform.</p>
                    <button class="btn congratulation-btn" onclick="closeCongratulation()">
                        Continue to Dashboard
                    </button>
                </div>
                
                <?php for($i = 0; $i < 100; $i++): ?>
                <div class="confetti" style="
                    left: <?= rand(0, 100) ?>%;
                    background-color: hsl(<?= rand(0, 360) ?>, 100%, 50%);
                    animation-delay: <?= rand(0, 5) ?>s;
                    width: <?= rand(5, 10) ?>px;
                    height: <?= rand(5, 10) ?>px;
                "></div>
                <?php endfor; ?>
            </div>
        </div>
        <?php 
            $_SESSION['show_congratulation'] = false;
            unset($_SESSION['show_congratulation']);
        endif; ?>
        
        <!-- ID Review Modal -->
        <div class="id-review-modal" id="idReviewModal">
            <div class="id-review-content">
                <span class="id-review-close" onclick="closeIdReview()">&times;</span>
                <h3 class="id-review-title">ID Card Review</h3>
                <div class="id-review-images">
                    <div class="id-review-image-container">
                        <img src="<?= $nid_front ?>" alt="NID Front" class="id-review-image">
                        <div class="id-review-caption">National ID Front Side</div>
                    </div>
                    <div class="id-review-image-container">
                        <img src="<?= $nid_back ?>" alt="NID Back" class="id-review-image">
                        <div class="id-review-caption">National ID Back Side</div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Camera Modal -->
        <div class="camera-modal" id="cameraModal">
            <div class="camera-content">
                <span class="camera-close" onclick="closeCameraModal()">&times;</span>
                <h3 class="camera-title" id="cameraModalTitle">Take Photo</h3>
                <video id="cameraPreview" class="camera-preview" autoplay playsinline></video>
                <canvas id="cameraCanvas" style="display:none;"></canvas>
                <div class="camera-buttons">
                    <button class="camera-button camera-switch" id="switchCameraBtn">
                        <i class="fas fa-sync-alt"></i> Switch Camera
                    </button>
                    <button class="camera-button camera-capture" id="captureBtn">
                        <i class="fas fa-camera"></i> Capture
                    </button>
                    <button class="camera-button camera-cancel" onclick="closeCameraModal()">
                        <i class="fas fa-times"></i> Cancel
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Profile Header with Particles.js -->
        <div class="profile-header text-center">
            <div id="particles-js"></div>
            <div class="profile-header-content">
                <div class="profile-pic-container">
                    <img src="<?= $profile_picture ?>" alt="Profile Picture" class="profile-pic">
                </div>
                <h3><?= htmlspecialchars($user['username']) ?></h3>
                <p class="mb-0">
                    <i class="fas fa-phone-alt me-2"></i><?= htmlspecialchars($user['mobile_number']) ?>
                    <span class="mx-2">•</span>
                    <i class="fas fa-calendar-alt me-2"></i>Member since <?= date('M Y', strtotime($user['created_at'])) ?>
                </p>
                
                <!-- Verification Badge -->
                <div class="verification-badge">
                    <?php if($user['is_verified']): ?>
                        <i class="fas fa-check-circle verified-badge" title="Verified Account"></i>
                    <?php else: ?>
                        <i class="fas fa-times-circle not-verified-badge" title="Not Verified"></i>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <!-- Verification Progress -->
        <div class="user-info-card">
            <h5 class="mb-4"><i class="fas fa-tasks me-2"></i>Verification Progress</h5>
            
            <?php
            $total_steps = 3;
            $completed_steps = 0;
            if(!empty($user['profile_picture']) && $user['profile_picture'] !== $default_profile_pic) $completed_steps++;
            if(!empty($user['nid_front'])) $completed_steps++;
            if(!empty($user['nid_back'])) $completed_steps++;
            $progress = ($completed_steps / $total_steps) * 100;
            ?>
            
            <div class="progress-container">
                <div class="progress-bar">
                    <div class="progress" style="width: <?= $progress ?>%"></div>
                </div>
                <div class="progress-text"><?= round($progress) ?>% Complete</div>
            </div>
            
            <div class="document-status">
                <div class="document-item">
                    <div class="document-icon <?= (!empty($user['profile_picture']) && $user['profile_picture'] !== $default_profile_pic) ? 'completed' : 'pending' ?>">
                        <i class="fas <?= (!empty($user['profile_picture']) && $user['profile_picture'] !== $default_profile_pic) ? 'fa-check-circle' : 'fa-times-circle' ?>"></i>
                    </div>
                    <div>Profile Picture</div>
                </div>
                <div class="document-item">
                    <div class="document-icon <?= !empty($user['nid_front']) ? 'completed' : 'pending' ?>">
                        <i class="fas <?= !empty($user['nid_front']) ? 'fa-check-circle' : 'fa-times-circle' ?>"></i>
                    </div>
                    <div>NID Front</div>
                </div>
                <div class="document-item">
                    <div class="document-icon <?= !empty($user['nid_back']) ? 'completed' : 'pending' ?>">
                        <i class="fas <?= !empty($user['nid_back']) ? 'fa-check-circle' : 'fa-times-circle' ?>"></i>
                    </div>
                    <div>NID Back</div>
                </div>
            </div>
            
            <!-- Review ID Button (only show if both NID sides are uploaded) -->
            <?php if(!empty($user['nid_front']) && !empty($user['nid_back'])): ?>
            <div class="text-center mt-4">
                <button class="btn btn-primary" onclick="openIdReview()">
                    <i class="fas fa-id-card me-2"></i>Review My ID Card
                </button>
            </div>
            <?php endif; ?>
        </div>
        
        <div class="row">
            <!-- Left Column -->
            <div class="col-md-6">
                <!-- Profile Picture Upload (Only show if still using default) -->
                <div class="document-card <?= (!empty($user['profile_picture']) && $user['profile_picture'] !== $default_profile_pic) ? 'completed' : '' ?>">
                    <h5 class="mb-4"><i class="fas fa-camera me-2"></i>Profile Picture</h5>
                    <img src="<?= $profile_picture ?>" alt="Profile Picture Preview" class="document-image" id="profilePicPreview">
                    
                    <?php if($user['profile_picture'] === $default_profile_pic): ?>
                    <form method="post" enctype="multipart/form-data" id="profilePicForm">

                        <div class="mb-3" id="profile_picture_file_input">
                            <label class="upload-label">Upload Profile Picture (Max 2MB)</label>
                            <div class="file-input-container">
                                <div class="file-input-button">
                                    <i class="fas fa-upload me-2"></i> Choose File
                                </div>
                                <input type="file" class="file-input" name="profile_picture" accept="image/*" onchange="previewImage(this, 'profilePicPreview')">
                            </div>
                            <p class="upload-status">Only JPG, PNG, JPEG, GIF files are allowed</p>
                        </div>
                        
                        <input type="hidden" name="camera_image_data" id="profile_picture_camera_data">
                        <button type="submit" class="btn-upload">
                            <i class="fas fa-upload me-2"></i>Upload Profile Picture
                        </button>
                    </form>
                    <?php else: ?>
                    <div class="permanent-message">
                        <i class="fas fa-lock me-2 text-success"></i> 
                        <strong>Picture Uploaded</strong>
                        <p class="mb-0 mt-2">Your profile picture has been successfully uploaded and cannot be changed.</p>
                    </div>
                    <?php endif; ?>
                </div>
                
                <!-- User Information -->
                <div class="user-info-card">
                    <h5 class="mb-4"><i class="fas fa-user-circle me-2"></i>Personal Information</h5>
                    <div class="info-item">
                        <div class="info-label">Username</div>
                        <div class="info-value"><?= htmlspecialchars($user['username']) ?></div>
                    </div>
                    <div class="info-item">
                        <div class="info-label">Mobile Number</div>
                        <div class="info-value"><?= htmlspecialchars($user['mobile_number']) ?></div>
                    </div>
                    <div class="info-item">
                        <div class="info-label">Account Status</div>
                        <div class="info-value">
                            <?php if($user['is_verified']): ?>
                                <span class="badge bg-success"><i class="fas fa-check-circle me-1"></i>Verified</span>
                            <?php else: ?>
                                <span class="badge bg-warning text-dark"><i class="fas fa-exclamation-circle me-1"></i>Pending Verification</span>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="info-item">
                        <div class="info-label">Member Since</div>
                        <div class="info-value"><?= date('M d, Y', strtotime($user['created_at'])) ?></div>
                    </div>
                </div>
            </div>
            
            <!-- Right Column -->
            <div class="col-md-6">
                <!-- NID Front Upload -->
                <div class="document-card <?= !empty($user['nid_front']) ? 'completed' : '' ?>">
                    <h5 class="mb-4"><i class="fas fa-id-card me-2"></i>NID Front Side</h5>
                    <img src="<?= $nid_front ?>" alt="NID Front" class="document-image" id="nidFrontPreview">
                    
                    <?php if(empty($user['nid_front'])): ?>
                    <form method="post" enctype="multipart/form-data" id="nidFrontForm">
                        
                        <div class="mb-3" id="nid_front_file_input">
                            <label class="upload-label">Upload NID Front (Max 2MB)</label>
                            <div class="file-input-container">
                                <div class="file-input-button">
                                    <i class="fas fa-upload me-2"></i> Choose File
                                </div>
                                <input type="file" class="file-input" name="nid_front" accept="image/*" onchange="previewImage(this, 'nidFrontPreview')">
                            </div>
                            <p class="upload-status">Only JPG, PNG, JPEG files are allowed</p>
                        </div>
                        
                        <input type="hidden" name="camera_image_data" id="nid_front_camera_data">
                        <button type="submit" class="btn-upload">
                            <i class="fas fa-upload me-2"></i>Upload NID Front
                        </button>
                    </form>
                    <?php else: ?>
                    <div class="alert alert-success mt-3">
                        <i class="fas fa-check-circle me-2"></i> NID Front has been uploaded and verified
                    </div>
                    <?php endif; ?>
                </div>
                
                <!-- NID Back Upload -->
                <div class="document-card <?= !empty($user['nid_back']) ? 'completed' : '' ?>">
                    <h5 class="mb-4"><i class="fas fa-id-card me-2"></i>NID Back Side</h5>
                    <img src="<?= $nid_back ?>" alt="NID Back" class="document-image" id="nidBackPreview">
                    
                    <?php if(empty($user['nid_back'])): ?>
                    <form method="post" enctype="multipart/form-data" id="nidBackForm">
                        
                        <div class="mb-3" id="nid_back_file_input">
                            <label class="upload-label">Upload NID Back (Max 2MB)</label>
                            <div class="file-input-container">
                                <div class="file-input-button">
                                    <i class="fas fa-upload me-2"></i> Choose File
                                </div>
                                <input type="file" class="file-input" name="nid_back" accept="image/*" onchange="previewImage(this, 'nidBackPreview')">
                            </div>
                            <p class="upload-status">Only JPG, PNG, JPEG files are allowed</p>
                        </div>
                        
                        <input type="hidden" name="camera_image_data" id="nid_back_camera_data">
                        <button type="submit" class="btn-upload">
                            <i class="fas fa-upload me-2"></i>Upload NID Back
                        </button>
                    </form>
                    <?php else: ?>
                    <div class="alert alert-success mt-3">
                        <i class="fas fa-check-circle me-2"></i> NID Back has been uploaded and verified
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <style>
    body {
        margin: 0;
        padding: 0;
        overflow-x: hidden;
      }
  
      .footer {
        position: fixed;
        bottom: 0;
        width: 100%;
        background: rgba(15, 32, 39, 0.95);
        backdrop-filter: blur(4px);
        color: white;
        padding: 10px 0;
        box-shadow: 0 -4px 20px rgba(0, 0, 0, 0.4);
        z-index: 999;
        transition: background 0.3s ease;
      }
  
      .footer.scrolled {
        background: rgba(15, 32, 39, 0.8);
      }
  
      .footer .row {
        display: flex;
        justify-content: space-around;
        align-items: center;
        text-align: center;
        max-width: 1000px;
        margin: 0 auto;
        padding: 0 10px;
        flex-wrap: wrap;
        box-sizing: border-box;
        overflow-x: hidden;
      }
  
      .footer .col {
        flex: 1 1 22%;
        min-width: 60px;
        max-width: 25%;
        margin: 5px 0;
        box-sizing: border-box;
      }
  
      .footer .col a {
        display: flex;
        flex-direction: column;
        align-items: center;
        text-decoration: none;
        color: white;
        transition: transform 0.3s;
      }
  
      .footer .col a:hover {
        transform: scale(1.1);
      }
  
      .footer-icon {
        width: 100%;
        max-width: 48px;
        height: auto;
        object-fit: contain;
        transition: all 0.3s ease-in-out;
        filter: drop-shadow(0 4px 8px rgba(0,0,0,0.4));
      }
  
      .footer p {
        margin: 4px 0 0;
        font-size: 14px;
      }
  
      .footer .col:nth-child(1) img {
        animation: float1 4s ease-in-out infinite;
        filter: drop-shadow(0 5px 12px rgba(255, 111, 0, 0.4));
      }
  
      .footer .col:nth-child(2) img {
        animation: float2 5s ease-in-out infinite;
        filter: drop-shadow(0 5px 12px rgba(37, 211, 102, 0.5));
      }
  
      .footer .col:nth-child(3) img {
        animation: float3 4.5s ease-in-out infinite;
        filter: drop-shadow(0 5px 12px rgba(255, 255, 0, 0.5));
      }
  
      .footer .col:nth-child(4) img {
        animation: float4 5.2s ease-in-out infinite;
        filter: drop-shadow(0 5px 12px rgba(0, 123, 255, 0.5));
      }
  
      @keyframes float1 {
        0%, 100% { transform: translateY(0px) rotateZ(0deg); }
        50% { transform: translateY(-4px) rotateZ(1deg); }
      }
  
      @keyframes float2 {
        0%, 100% { transform: translateY(0px) rotateZ(0deg); }
        50% { transform: translateY(-3px) rotateZ(-1deg); }
      }
  
      @keyframes float3 {
        0%, 100% { transform: translateY(0px) rotateZ(0deg); }
        50% { transform: translateY(-5px) rotateZ(1deg); }
      }
  
      @keyframes float4 {
        0%, 100% { transform: translateY(0px) rotateZ(0deg); }
        50% { transform: translateY(-3px) rotateZ(-1.5deg); }
      }
  
      @media (max-width: 768px) {
        .footer-icon {
          max-width: 38px;
        }
        .footer p {
          font-size: 12px;
        }
      }
  
      @media (max-width: 480px) {
        .footer .row {
          flex-direction: row;
          justify-content: space-between;
          padding: 0 10px;
        }
        .footer-icon {
          max-width: 32px;
        }
        .footer p {
          font-size: 11px;
        }
        .footer .col {
          flex: 1 1 22%;
        }
      }
    </style>
    
    <!-- Footer Navigation -->
    <div class="footer">
        <div class="row">
            <div class="col">
                <a href="dashboard.php">
                    <img class="footer-icon" src="images/footer/home.png" alt="Home">
                    <p>Home</p>
                </a>
            </div>
            <div class="col">
                <a href="https://wa.me/<?= $whatsapp_number ?>" target="_blank">
                    <img class="footer-icon" src="images/footer/whatsapp.png" alt="WhatsApp">
                    <p>WhatsApp</p>
                </a>
            </div>
            <div class="col">
                <a href="profile.php">
                    <img class="footer-icon" src="images/footer/user-4.png" alt="Profile">
                    <p>Profile</p>
                </a>
            </div>
            <div class="col">
                <a href="<?= $_SESSION["logged_in"] ? 'logout.php' : 'login.php' ?>">
                    <img class="footer-icon" src="images/footer/<?= $_SESSION["logged_in"] ? 'logout-2.png' : 'login.png' ?>" alt="Login">
                    <p><?= $_SESSION["logged_in"] ? 'LogOut' : 'Login' ?></p>
                </a>
            </div>
        </div>
    </div>
    <br><br><br><br>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/particles.js/2.0.0/particles.min.js"></script>
    <script>
        // Initialize particles.js
        document.addEventListener('DOMContentLoaded', function() {
            particlesJS('particles-js', {
                "particles": {
                    "number": {
                        "value": 80,
                        "density": {
                            "enable": true,
                            "value_area": 800
                        }
                    },
                    "color": {
                        "value": "#ffffff"
                    },
                    "shape": {
                        "type": "circle",
                        "stroke": {
                            "width": 0,
                            "color": "#000000"
                        },
                        "polygon": {
                            "nb_sides": 5
                        }
                    },
                    "opacity": {
                        "value": 0.5,
                        "random": false,
                        "anim": {
                            "enable": false,
                            "speed": 1,
                            "opacity_min": 0.1,
                            "sync": false
                        }
                    },
                    "size": {
                        "value": 3,
                        "random": true,
                        "anim": {
                            "enable": false,
                            "speed": 40,
                            "size_min": 0.1,
                            "sync": false
                        }
                    },
                    "line_linked": {
                        "enable": true,
                        "distance": 150,
                        "color": "#ffffff",
                        "opacity": 0.4,
                        "width": 1
                    },
                    "move": {
                        "enable": true,
                        "speed": 2,
                        "direction": "none",
                        "random": false,
                        "straight": false,
                        "out_mode": "out",
                        "bounce": false,
                        "attract": {
                            "enable": false,
                            "rotateX": 600,
                            "rotateY": 1200
                        }
                    }
                },
                "interactivity": {
                    "detect_on": "canvas",
                    "events": {
                        "onhover": {
                            "enable": true,
                            "mode": "grab"
                        },
                        "onclick": {
                            "enable": true,
                            "mode": "push"
                        },
                        "resize": true
                    },
                    "modes": {
                        "grab": {
                            "distance": 140,
                            "line_linked": {
                                "opacity": 1
                            }
                        },
                        "bubble": {
                            "distance": 400,
                            "size": 40,
                            "duration": 2,
                            "opacity": 8,
                            "speed": 3
                        },
                        "repulse": {
                            "distance": 200,
                            "duration": 0.4
                        },
                        "push": {
                            "particles_nb": 4
                        },
                        "remove": {
                            "particles_nb": 2
                        }
                    }
                },
                "retina_detect": true
            });
        });

        // Auto-dismiss alerts after 5 seconds
        setTimeout(() => {
            const alerts = document.querySelectorAll('.alert');
            alerts.forEach(alert => {
                new bootstrap.Alert(alert).close();
            });
        }, 5000);
        
        // Preview image before upload
        function previewImage(input, previewId) {
            if (input.files && input.files[0]) {
                const reader = new FileReader();
                const imgElement = document.getElementById(previewId);
                
                reader.onload = function(e) {
                    imgElement.src = e.target.result;
                };
                
                reader.readAsDataURL(input.files[0]);
            }
        }
        
        // Close congratulation modal
        function closeCongratulation() {
            document.querySelector('.congratulation-modal').style.display = 'none';
        }
        
        // Create confetti effect
        function createConfetti() {
            const container = document.querySelector('.congratulation-content');
            for (let i = 0; i < 100; i++) {
                const confetti = document.createElement('div');
                confetti.className = 'confetti';
                confetti.style.left = Math.random() * 100 + '%';
                confetti.style.backgroundColor = `hsl(${Math.random() * 360}, 100%, 50%)`;
                confetti.style.animationDelay = Math.random() * 5 + 's';
                confetti.style.width = Math.random() * 5 + 5 + 'px';
                confetti.style.height = Math.random() * 5 + 5 + 'px';
                container.appendChild(confetti);
            }
        }
        
        // Initialize confetti if congratulation modal is shown
        if (document.querySelector('.congratulation-modal')) {
            createConfetti();
        }
        
        // ID Review Modal Functions
        function openIdReview() {
            document.getElementById('idReviewModal').style.display = 'flex';
            document.body.style.overflow = 'hidden';
        }
        
        function closeIdReview() {
            document.getElementById('idReviewModal').style.display = 'none';
            document.body.style.overflow = 'auto';
        }
        
        // Close modal when clicking outside
        window.onclick = function(event) {
            const modal = document.getElementById('idReviewModal');
            if (event.target == modal) {
                closeIdReview();
            }
        }
        
        // Camera functionality
        let currentStream = null;
        let currentCameraType = 'user'; // 'user' for front camera, 'environment' for back camera
        let currentUploadType = ''; // To track which form is using the camera
        
        function openCamera(uploadType) {
            currentUploadType = uploadType;
            document.getElementById('cameraModalTitle').textContent = `Take ${uploadType.replace('_', ' ')} Photo`;
            document.getElementById('cameraModal').style.display = 'flex';
            document.body.style.overflow = 'hidden';
            
            // Set active button in upload options
            document.querySelectorAll('.upload-option-btn').forEach(btn => {
                btn.classList.remove('active');
            });
            event.target.classList.add('active');
            
            startCamera();
        }
        
        function closeCameraModal() {
            document.getElementById('cameraModal').style.display = 'none';
            document.body.style.overflow = 'auto';
            stopCamera();
        }
        
        function startCamera() {
            stopCamera(); // Stop any existing stream
            
            const constraints = {
                video: {
                    facingMode: currentCameraType,
                    width: { ideal: 1280 },
                    height: { ideal: 720 }
                },
                audio: false
            };
            
            const video = document.getElementById('cameraPreview');
            
            navigator.mediaDevices.getUserMedia(constraints)
                .then(function(stream) {
                    currentStream = stream;
                    video.srcObject = stream;
                })
                .catch(function(err) {
                    console.error("Error accessing camera: ", err);
                    alert("Could not access the camera. Please make sure you have granted camera permissions.");
                });
        }
        
        function stopCamera() {
            if (currentStream) {
                currentStream.getTracks().forEach(track => track.stop());
                currentStream = null;
            }
        }
        
        function switchCamera() {
            currentCameraType = currentCameraType === 'user' ? 'environment' : 'user';
            startCamera();
        }
        
        document.getElementById('switchCameraBtn').addEventListener('click', switchCamera);
        
        document.getElementById('captureBtn').addEventListener('click', function() {
            const video = document.getElementById('cameraPreview');
            const canvas = document.getElementById('cameraCanvas');
            const context = canvas.getContext('2d');
            
            // Set canvas dimensions to match video stream
            canvas.width = video.videoWidth;
            canvas.height = video.videoHeight;
            
            // Draw current video frame to canvas
            context.drawImage(video, 0, 0, canvas.width, canvas.height);
            
            // Get image data URL
            const imageDataUrl = canvas.toDataURL('image/jpeg', 0.8);
            
            // Set the image data to the corresponding hidden input
            document.getElementById(`${currentUploadType}_camera_data`).value = imageDataUrl;
            
            // Preview the captured image
            document.getElementById(`${currentUploadType}Preview`).src = imageDataUrl;
            
            // Hide the file input and show only camera data will be submitted
            document.getElementById(`${currentUploadType}_file_input`).style.display = 'none';
            
            // Close camera modal
            closeCameraModal();
        });
        
        function showFileInput(uploadType) {
            // Set active button in upload options
            document.querySelectorAll('.upload-option-btn').forEach(btn => {
                btn.classList.remove('active');
            });
            event.target.classList.add('active');
            
            // Show file input and clear camera data
            document.getElementById(`${uploadType}_file_input`).style.display = 'block';
            document.getElementById(`${uploadType}_camera_data`).value = '';
        }
        
        // Handle form submission with camera data
        document.querySelectorAll('form').forEach(form => {
            form.addEventListener('submit', function(e) {
                const uploadType = this.id.replace('Form', '');
                const cameraData = document.getElementById(`${uploadType}_camera_data`).value;
                
                if (cameraData) {
                    // If camera data exists, create a blob and append to form
                    const blob = dataURLtoBlob(cameraData);
                    const file = new File([blob], `${uploadType}_${new Date().getTime()}.jpg`, { type: 'image/jpeg' });
                    
                    const dataTransfer = new DataTransfer();
                    dataTransfer.items.add(file);
                    
                    // Find the file input and set the files
                    const fileInput = this.querySelector('input[type="file"]');
                    fileInput.files = dataTransfer.files;
                }
            });
        });
        
        function dataURLtoBlob(dataurl) {
            const arr = dataurl.split(',');
            const mime = arr[0].match(/:(.*?);/)[1];
            const bstr = atob(arr[1]);
            let n = bstr.length;
            const u8arr = new Uint8Array(n);
            
            while(n--) {
                u8arr[n] = bstr.charCodeAt(n);
            }
            
            return new Blob([u8arr], { type: mime });
        }
    </script>
</body>
</html>