<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Voice Auto-Play System</title>
    <style>
        /* Basic styling for error messages */
        body {
            margin: 0;
            padding: 0;
            font-family: Arial, sans-serif;
            background: #f0f0f0;
            color: #333;
        }
        
        .error-container {
            display: none;
            position: fixed;
            top: 20px;
            right: 20px;
            background: #ffebee;
            border-left: 4px solid #d32f2f;
            padding: 15px;
            border-radius: 4px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            max-width: 350px;
            z-index: 10000;
        }
        
        .error-title {
            font-weight: bold;
            color: #d32f2f;
            margin-bottom: 5px;
        }
        
        .error-message {
            font-size: 14px;
            line-height: 1.4;
        }
        
        .close-btn {
            position: absolute;
            top: 10px;
            right: 10px;
            background: none;
            border: none;
            font-size: 18px;
            cursor: pointer;
            color: #d32f2f;
        }
        
        /* Hidden audio player styles */
        .hidden-audio-player {
            position: absolute;
            width: 1px;
            height: 1px;
            padding: 0;
            margin: -1px;
            overflow: hidden;
            clip: rect(0, 0, 0, 0);
            white-space: nowrap;
            border: 0;
        }
        
        /* Loading indicator */
        .audio-loading {
            display: none;
            position: fixed;
            bottom: 20px;
            right: 20px;
            background: rgba(0,0,0,0.7);
            color: white;
            padding: 10px 15px;
            border-radius: 20px;
            font-size: 14px;
            z-index: 1000;
        }
        
        /* Status panel for debugging */
        .status-panel {
            display: none;
            position: fixed;
            bottom: 20px;
            left: 20px;
            background: rgba(0,0,0,0.8);
            color: white;
            padding: 15px;
            border-radius: 8px;
            font-size: 14px;
            max-width: 300px;
            z-index: 1000;
        }
    </style>
</head>
<body>
    <!-- Hidden audio element -->
    <audio id="pageAudio" class="hidden-audio-player">
        <source src="images/Audio/login_page_audio.mp3" type="audio/mpeg">
        Your browser does not support the audio element.
    </audio>
    
    <div id="audioLoading" class="audio-loading">Loading audio...</div>
    
    <!-- Error display container -->
    <div id="errorContainer" class="error-container">
        <button class="close-btn" id="closeError">&times;</button>
        <div class="error-title" id="errorTitle">Audio Playback Error</div>
        <div class="error-message" id="errorMessage"></div>
    </div>
    
    <!-- Status panel for debugging (optional) -->
    <div id="statusPanel" class="status-panel">
        <div><strong>Audio Status:</strong> <span id="audioStatus">Checking...</span></div>
        <div><strong>Last Played:</strong> <span id="lastPlayed">Never</span></div>
    </div>

    <script>
        (function() {
            const audio = document.getElementById('pageAudio');
            const loadingIndicator = document.getElementById('audioLoading');
            const errorContainer = document.getElementById('errorContainer');
            const errorTitle = document.getElementById('errorTitle');
            const errorMessage = document.getElementById('errorMessage');
            const closeErrorBtn = document.getElementById('closeError');
            const statusPanel = document.getElementById('statusPanel');
            const audioStatus = document.getElementById('audioStatus');
            const lastPlayed = document.getElementById('lastPlayed');
            
            const STORAGE_KEY = 'lastAudioPlayTime';
            const TWELVE_HOURS = 12 * 60 * 60 * 1000; // 12 hours in milliseconds
            
            // For debugging purposes, you can show the status panel by adding ?debug=true to URL
            const urlParams = new URLSearchParams(window.location.search);
            if (urlParams.get('debug') === 'true') {
                statusPanel.style.display = 'block';
            }
            
            // Function to show error message
            function showError(title, message) {
                errorTitle.textContent = title;
                errorMessage.textContent = message;
                errorContainer.style.display = 'block';
                
                // Auto-hide error after 10 seconds
                setTimeout(() => {
                    errorContainer.style.display = 'none';
                }, 10000);
            }
            
            // Close error button event
            closeErrorBtn.addEventListener('click', function() {
                errorContainer.style.display = 'none';
            });
            
            // Update status panel
            function updateStatus(status) {
                if (urlParams.get('debug') === 'true') {
                    audioStatus.textContent = status;
                    
                    const lastPlayTime = localStorage.getItem(STORAGE_KEY);
                    if (lastPlayTime) {
                        const date = new Date(parseInt(lastPlayTime));
                        lastPlayed.textContent = date.toLocaleString();
                    } else {
                        lastPlayed.textContent = 'Never';
                    }
                }
            }
            
            // Function to check if audio should play
            function shouldPlayAudio() {
                const lastPlayTime = localStorage.getItem(STORAGE_KEY);
                
                if (!lastPlayTime) {
                    updateStatus('First time play');
                    return true;
                }
                
                const currentTime = new Date().getTime();
                const elapsedTime = currentTime - parseInt(lastPlayTime);
                
                if (elapsedTime >= TWELVE_HOURS) {
                    updateStatus('12 hours passed, can play');
                    return true;
                } else {
                    updateStatus('Already played within 12 hours');
                    const remainingTime = TWELVE_HOURS - elapsedTime;
                    const hours = Math.floor(remainingTime / (1000 * 60 * 60));
                    console.log(`Audio already played. Next available in ${hours} hours.`);
                    return false;
                }
            }
            
            // Function to play audio
            function playAudio() {
                if (!shouldPlayAudio()) {
                    return;
                }
                
                // Show loading indicator
                if (loadingIndicator) loadingIndicator.style.display = 'block';
                updateStatus('Loading audio...');
                
                // Play audio
                audio.play().then(() => {
                    // Store the playback time
                    const currentTime = new Date().getTime();
                    localStorage.setItem(STORAGE_KEY, currentTime.toString());
                    
                    console.log('Audio playback started');
                    updateStatus('Playing audio');
                    
                    // Hide loading indicator when audio ends
                    audio.addEventListener('ended', function() {
                        if (loadingIndicator) loadingIndicator.style.display = 'none';
                        console.log('Audio playback completed');
                        updateStatus('Playback completed');
                    });
                    
                }).catch(error => {
                    console.error('Audio playback failed:', error);
                    if (loadingIndicator) loadingIndicator.style.display = 'none';
                    
                    // Show appropriate error message
                    let errorMsg = 'Unknown error occurred';
                    if (error.name === 'NotAllowedError') {
                        errorMsg = 'Autoplay was blocked by the browser. Please click anywhere on the page to play audio.';
                    } else if (error.name === 'NotSupportedError') {
                        errorMsg = 'Audio format is not supported by your browser.';
                    } else if (error.name === 'AbortError') {
                        errorMsg = 'Audio playback was aborted.';
                    }
                    
                    showError('Playback Error', errorMsg);
                    updateStatus('Error: ' + error.name);
                });
            }
            
            // Handle audio errors
            audio.addEventListener('error', function(e) {
                if (loadingIndicator) loadingIndicator.style.display = 'none';
                console.error('Error occurred during audio playback:', audio.error);
                
                let errorMsg = 'Failed to load audio. ';
                switch(audio.error.code) {
                    case MediaError.MEDIA_ERR_ABORTED:
                        errorMsg += 'Playback was aborted.';
                        break;
                    case MediaError.MEDIA_ERR_NETWORK:
                        errorMsg += 'Network error occurred.';
                        break;
                    case MediaError.MEDIA_ERR_DECODE:
                        errorMsg += 'Audio decoding failed.';
                        break;
                    case MediaError.MEDIA_ERR_SRC_NOT_SUPPORTED:
                        errorMsg += 'Audio format not supported.';
                        break;
                    default:
                        errorMsg += 'Unknown error occurred.';
                }
                
                showError('Audio Error', errorMsg);
                updateStatus('Error: ' + errorMsg);
            });
            
            // Check if audio should play on page load
            document.addEventListener('DOMContentLoaded', function() {
                updateStatus('Page loaded');
                playAudio();
            });
            
            // Fallback for browsers that don't support autoplay
            function playAudioOnInteraction() {
                if (shouldPlayAudio()) {
                    playAudio();
                }
            }
            
            // Add click event listener to document as fallback
            document.addEventListener('click', function() {
                if (audio.paused && shouldPlayAudio()) {
                    playAudioOnInteraction();
                }
            }, { once: true });
            
            // Also try to play on keypress for accessibility
            document.addEventListener('keydown', function() {
                if (audio.paused && shouldPlayAudio()) {
                    playAudioOnInteraction();
                }
            }, { once: true });
        })();
    </script>
</body>
</html>